// server.js
const path = require("path");
const express = require("express");
const http = require("http");
const https = require("https");
const WebSocket = require("ws");
const os = require("os");
const multer = require("multer");
const FormData = require("form-data");
const upload = multer({ dest: "uploads/" });
const fs = require("fs");
const cors = require("cors");
const axios = require("axios");

const app = express();

// Kiểm tra môi trường và tạo server tương ứng
const isProduction =
  process.env.NODE_ENV === "production" || process.env.HTTPS === "true";
let server;
let wss;

if (isProduction) {
  // HTTPS cho production
  try {
    const httpsOptions = {
      key: fs.readFileSync(
        "/etc/letsencrypt/live/remote.bigscore.vn/privkey.pem"
      ),
      cert: fs.readFileSync(
        "/etc/letsencrypt/live/remote.bigscore.vn/fullchain.pem"
      ),
    };
    server = https.createServer(httpsOptions, app);
    console.log("🔒 HTTPS server initialized for production");
  } catch (error) {
    console.error("❌ Error loading SSL certificates:", error.message);
    console.log("⚠️  Falling back to HTTP server");
    server = http.createServer(app);
  }
} else {
  // HTTP cho development
  server = http.createServer(app);
  console.log("🔓 HTTP server initialized for development");
}

wss = new WebSocket.Server({ server });

// Quản lý client theo room
const rooms = {}; // { roomId: Set<ws> }

const keyRooms = {}; // Quản lý theo key riêng

// Import Firebase Admin SDK
const admin = require("firebase-admin");
const { getMessaging } = require("firebase-admin/messaging");
const serviceAccount = require("./serviceAccountKey.json");
const { error } = require("console");

admin.initializeApp({
  credential: admin.credential.cert(serviceAccount),
});

const roomNotifies = {};
// Lấy IP LAN
function getLocalIP() {
  const interfaces = os.networkInterfaces();
  for (const name of Object.keys(interfaces)) {
    for (const iface of interfaces[name]) {
      if (iface.family === "IPv4" && !iface.internal) return iface.address;
    }
  }
  return "0.0.0.0";
}

function findAppByRoom(roomId) {
  if (!rooms[roomId]) return null;
  // Giả sử device = "app" thì mới trả về
  for (const client of rooms[roomId]) {
    if (client.device === "app") {
      return client.ws; // trả về socket luôn
    }
  }
  return null;
}

// Serve file HTML client
app.use(express.static(path.join(__dirname)));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(cors());

wss.on("connection", (ws) => {
  const now = new Date();
  const requestTime = now.toLocaleTimeString("vi-VN", {
    hour: "2-digit",
    minute: "2-digit",
    second: "2-digit",
  });

  let currentRoom = null;
  let clientId = "";
  let currentKey = null;

  async function sendFCMToTokens(tokens, key, payloadData) {
    if (!tokens || tokens.length === 0) return;

    const BATCH = 500;
    for (let i = 0; i < tokens.length; i += BATCH) {
      const batch = tokens.slice(i, i + BATCH);

      const message = {
        tokens: batch,
        notification: {
          title: "Có yêu cầu nhân viên",
          body: `${payloadData.requestStaff} gọi lúc ${payloadData.requestTime}`,
        },
        data: {
          screen: String(payloadData.screen || ""),
          idDevice: String(payloadData.idDevice || ""),
          requestTime: String(payloadData.requestTime || ""),
          requestStaff: String(payloadData.requestStaff || ""),
          key: String(key || ""),
        },
      };

      try {
        const response = await getMessaging().sendEachForMulticast(message);
        if (response.failureCount > 0) {
        }
      } catch (err) {
        console.error("sendMulticast error:", err);
      }
    }
  }
  ws.on("message", (message) => {
    try {
      const data = JSON.parse(message.toString());
      console.log("data", data);
      clientId = data.clientId;
      if (data.type === "ping") {
        // Trả về pong ngay lập tức
        ws.send(
          JSON.stringify({
            type: "pong",
            timestamp: data.timestamp,
          })
        );
        return; // Không xử lý tiếp
      }
      switch (data.type) {
        case "join-room": {
          const roomId = data.room;
          currentRoom = roomId;

          if (!rooms[roomId]) {
            rooms[roomId] = new Set();
          }

          // Kiểm tra xem client đã tồn tại chưa
          const alreadyInRoom = Array.from(rooms[roomId]).some(
            (c) => c.id === clientId
          );
          if (alreadyInRoom) {
            ws.send(
              JSON.stringify({
                type: "error",
                msg: "Bạn đã có trong phòng, không thể join thêm lần nữa",
              })
            );
            return; // không thêm vào room
          }

          const clientsInRoomOld = Array.from(rooms[roomId]).map((c) => c.id);

          if (
            (clientsInRoomOld.length == 0 && data.device === "app") ||
            (clientsInRoomOld.length > 0 && data.device === "web")
          ) {
            rooms[roomId].add({
              ws,
              id: clientId,
              device: data.device,
              matchType: data.matchType, // ✅ Lưu matchType từ client
            });
          } else {
            ws.send(
              JSON.stringify({
                type: "error",
                room: roomId,
                msg: `Room ${roomId} chưa tồn tại`,
              })
            );
            return;
          }

          // Tìm matchType từ app client (nếu có)
          let matchTypeToSend = 0; // Mặc định là 0
          const appClient = Array.from(rooms[roomId]).find(
            (client) => client.device === "app"
          );
          if (appClient && appClient.matchType) {
            matchTypeToSend = appClient.matchType;
          }

          // Hiển thị danh sách client trong room
          const clientsInRoomNew = Array.from(rooms[roomId]).map((c) => c.id);

          ws.send(
            JSON.stringify({
              type: "room-joined",
              room: roomId,
              clientId: clientId,
              msg: `Bạn đã vào phòng ${roomId}`,
              clients: clientsInRoomNew,
              matchType: matchTypeToSend, // ✅ Gửi matchType về client
            })
          );
          break;
        }
        case "join-league": {
          const roomId = data.room;
          currentRoom = roomId;

          // Tạo room nếu chưa tồn tại
          if (!rooms[roomId]) {
            rooms[roomId] = new Set();
          }

          // Kiểm tra xem client đã tồn tại chưa
          const alreadyInRoom = Array.from(rooms[roomId]).some(
            (c) => c.id === clientId
          );
          if (alreadyInRoom) {
            ws.send(
              JSON.stringify({
                type: "error",
                msg: "Bạn đã có trong phòng, không thể join thêm lần nữa",
              })
            );
            return; // không thêm vào room
          }

          // Thêm client vào room (không có ràng buộc device type)
          rooms[roomId].add({
            ws,
            id: clientId,
            device: data.device,
            matchType: data.matchType,
          });

          console.log(`👤 Client join league room:`, {
            clientId: clientId,
            roomId: roomId,
            device: data.device,
            matchType: data.matchType,
            timestamp: new Date().toLocaleTimeString("vi-VN"),
            totalClientsInRoom: rooms[roomId].size,
            allClientsInRoom: Array.from(rooms[roomId]).map((c) => c.id),
          });

          // Tìm matchType từ app client (nếu có)
          let matchTypeToSend = 0; // Mặc định là 0
          const appClient = Array.from(rooms[roomId]).find(
            (client) => client.device === "app"
          );
          if (appClient && appClient.matchType) {
            matchTypeToSend = appClient.matchType;
          }

          // Hiển thị danh sách client trong room
          const clientsInRoomNew = Array.from(rooms[roomId]).map((c) => c.id);

          ws.send(
            JSON.stringify({
              type: "league-room-joined",
              room: roomId,
              clientId: clientId,
              msg: `Bạn đã vào phòng league ${roomId}`,
              clients: clientsInRoomNew,
              matchType: matchTypeToSend,
            })
          );
          break;
        }

        case "join-room-staff": {
          const { key, fcmToken } = data;
          currentKey = data.key;
          if (!key) {
            ws.send(
              JSON.stringify({ type: "error", msg: "Key không được để trống" })
            );
            return;
          }

          if (!keyRooms[key]) keyRooms[key] = new Set();

          // Kiểm tra client đã join chưa
          const existingClient = Array.from(keyRooms[key]).find(
            (c) => c.id === clientId
          );
          if (existingClient) {
            // Nếu client F5 → cập nhật socket mới
            existingClient.ws = ws;
            existingClient.fcmToken = fcmToken;
          } else {
            // Thêm client mới
            keyRooms[key].add({
              ws,
              id: clientId,
              device: data.device,
              fcmToken,
            });
          }

          if (!roomNotifies[key]) roomNotifies[key] = [];

          // Gửi danh sách notify cũ về cho client mới join
          const pending = roomNotifies[key].filter(
            (r) => r.status === "pending"
          );
          for (const notify of pending) {
            ws.send(
              JSON.stringify({
                type: "control",
                action: "call_staff",
                data: notify,
                key,
              })
            );
          }

          const clientsInKeyRoom = Array.from(keyRooms[key]).map((c) => c.id);
          ws.send(
            JSON.stringify({
              type: "room-joined-staff",
              key: currentKey,
              clientId,
              msg: `Staff đã vào key ${key}`,
              clients: clientsInKeyRoom,
              notifies: roomNotifies[key],
            })
          );

          break;
        }

        case "send-data": {
          const action = data.payload && data.payload.action;
          const payloadData = data.payload && data.payload.data;
          const key = data.key;

          // Xử lý action cụ thể
          if (action === "app_cancel_call_staff") {
            const result = payloadData;
            if (!key) break;
            if (!roomNotifies[key]) roomNotifies[key] = [];

            // Tìm request theo requestId hoặc uuid
            const notifyIndex = roomNotifies[key].findIndex(
              (n) => n.uuid === result.uuid
            );

            if (notifyIndex !== -1) {
              // Cập nhật trạng thái canceled
              roomNotifies[key][notifyIndex].status = "canceled";
              roomNotifies[key][notifyIndex].isCalling = false;
              roomNotifies[key][notifyIndex].canceledBy = result.canceledBy;
              roomNotifies[key][notifyIndex].canceledTime = result.canceledTime;
            }

            // Gửi thông tin hủy cho tất cả client trong phòng
            if (keyRooms[key] && keyRooms[key].size > 0) {
              for (const client of keyRooms[key]) {
                try {
                  if (client.ws && client.ws.readyState === WebSocket.OPEN) {
                    client.ws.send(
                      JSON.stringify({
                        type: "control",
                        action: "app_cancel_call_staff",
                        data: result,
                        key,
                      })
                    );
                  }
                } catch (err) {
                  console.error("WebSocket error:", err);
                }
              }
            }

            break;
          }

          if (action === "call_staff") {
            const result = payloadData;
            // đảm bảo key tồn tại
            if (!key) {
              break;
            }

            if (!roomNotifies[key]) roomNotifies[key] = [];

            // Thêm mới request vào đầu danh sách
            roomNotifies[key].unshift({
              screen: result.screen,
              idDevice: result.idDevice,
              requestStaff: result.requestStaff,
              requestTime: result.requestTime,
              requestId:
                result.requestId ||
                Date.now() + "-" + Math.random().toString(36).slice(2),
              acknowledged: false,
              handledBy: "",
              canceledBy: "",
              uuid: result.uuid,
              status: result.status,
              isCalling: result.isCalling,
            });

            const tokensSet = new Set();

            if (keyRooms[key] && keyRooms[key].size > 0) {
              // Duyệt từng client trong keyRooms[key]
              for (const client of keyRooms[key]) {
                try {
                  if (client.ws && client.ws.readyState === WebSocket.OPEN) {
                    // Người này online -> gửi websocket (đối tượng client đã là staff)
                    client.ws.send(
                      JSON.stringify({
                        type: "control",
                        action: "call_staff",
                        data: result,
                        key,
                      })
                    );
                  } else {
                    // Người này offline -> nếu có fcmToken thì gom lại để gửi push
                    if (client.fcmToken) tokensSet.add(client.fcmToken);
                  }
                } catch (err) {
                  // nếu lỗi socket, coi như offline -> collect token
                  if (client.fcmToken) tokensSet.add(client.fcmToken);
                }
              }
            } else {
            }

            const tokens = Array.from(tokensSet);
            if (tokens.length > 0) {
              // gửi FCM (không chặn luồng chính)
              sendFCMToTokens(tokens, key, result).catch((err) => {
                console.error("sendFCMToTokens error:", err);
              });
            }
            break;
          }
          if (
            action === "app_league_match_start" ||
            action === "app_league_match_score_update" ||
            action === "app_league_match_end"
          ) {
            console.log(`📢 Broadcasting ${action} to room ${currentRoom}`);

            if (!currentRoom) {
              console.warn("⚠️ No currentRoom for league action");
              return;
            }

            // Broadcast cho tất cả client trong room (trừ sender)
            if (rooms[currentRoom]) {
              rooms[currentRoom].forEach((client) => {
                if (
                  client.ws.readyState === WebSocket.OPEN &&
                  client.ws !== ws
                ) {
                  client.ws.send(
                    JSON.stringify({
                      type: "control",
                      action: action,
                      data: payloadData,
                    })
                  );
                  console.log(`✅ Sent ${action} to client ${client.id}`);
                }
              });
            }

            return; // ✅ Quan trọng: return để không chạy xuống phần broadcast chung bên dưới
          }

          // === PHẦN BROADCAST CHUNG CHO NHỮNG ACTION KHÁC ===
          if (data.key) {
            // Sử dụng currentKey để gửi cho những client trong cùng key (không lặp lại chính sender)
            if (currentKey && keyRooms[currentKey]) {
              keyRooms[currentKey].forEach((client) => {
                if (
                  client.ws.readyState === WebSocket.OPEN &&
                  client.ws !== ws
                ) {
                  client.ws.send(
                    JSON.stringify({
                      type: data.payload.type,
                      action: data.payload.action,
                      data: data.payload.data,
                      key: data.key,
                    })
                  );
                }
              });
            }
          } else {
            if (!currentRoom) return;
            rooms[currentRoom].forEach((client) => {
              if (client.ws.readyState === WebSocket.OPEN && client.ws !== ws) {
                client.ws.send(
                  JSON.stringify({
                    type: data.payload.type,
                    action: data.payload.action,
                    data: data.payload.data,
                  })
                );
              }
            });
          }

          break;
        }

        case "get-initial-data": {
          // Gửi yêu cầu sang app RN (client đã join room trước đó)

          const appClient = findAppByRoom(data.room); // hàm tìm socket của app trong room

          if (appClient) {
            // Lưu web client này lại để trả lời sau
            ws.pendingInitial = true;
            appClient.send(
              JSON.stringify({
                type: "request-initial-data",
                room: data.room,
                from: data.from,
              })
            );
          }
        }

        case "command": {
          // Handle different actions if needed
          switch (data.action) {
            case "staff_cancel": {
              const { key, idDevice, canceledBy, canceledTime, uuid, status } =
                data.data;
              if (!roomNotifies[key]) roomNotifies[key] = [];

              const req = roomNotifies[key].find(
                (r) =>
                  r.idDevice === idDevice &&
                  r.uuid === uuid &&
                  status === "canceled"
              );

              if (req) {
                req.canceledBy = canceledBy;
                req.canceledTime = canceledTime;
                req.status = status;
              }
              break;
            }

            case "staff_ack": {
              const { key, idDevice, handledBy, uuid, status } = data.data;

              if (!roomNotifies[key]) roomNotifies[key] = [];

              const req = roomNotifies[key].find(
                (r) =>
                  r.idDevice === idDevice && r.uuid === uuid && !r.handledBy
              );
              if (req) {
                req.acknowledged = true;
                req.handledBy = handledBy;
                req.timeHandle = new Date().toISOString();
                req.status = status;
              }
              break;
            }

            case "update_notifies": {
              const result = data.data;
              if (!roomNotifies[result.key]) roomNotifies[result.key] = [];

              // Kiểm tra đã tồn tại chưa
              const exists = roomNotifies[result.key].some(
                (r) => r.requestId === result.requestId
              );

              if (!exists) {
                roomNotifies[result.key].unshift(result);
              } else {
                // Nếu đã tồn tại thì chỉ update fields
                const req = roomNotifies[result.key].find(
                  (r) => r.requestId === result.requestId
                );
                Object.assign(req, result);
              }
              break;
            }
            // Add more cases for other actions if needed
            default:
              // Default action or do nothing
              break;
          }
          // broadcast theo key
          if (currentKey && keyRooms[currentKey]) {
            keyRooms[currentKey].forEach((client) => {
              if (client.ws !== ws && client.ws.readyState === WebSocket.OPEN) {
                client.ws.send(
                  JSON.stringify({
                    type: "control",
                    action: data.action,
                    data: data.data,
                    key: currentKey,
                  })
                );
              }
            });
          }

          if (!currentRoom) return;

          rooms[currentRoom].forEach((client) => {
            // Bỏ qua chính client đã gửi command
            if (client.ws !== ws && client.ws.readyState === WebSocket.OPEN) {
              client.ws.send(
                JSON.stringify({
                  type: "control",
                  action: data.action,
                  data: data.data,
                })
              );
            }
          });

          break;
        }

        case "initial-data": {
          const targetId = data.to; // nếu có "to" thì gửi cho 1 client
          const roomClients = rooms[currentRoom];

          if (roomClients) {
            if (targetId) {
              // gửi cho 1 client cụ thể
              const targetClient = Array.from(roomClients).find((c) =>
                c.id.endsWith(targetId)
              );
              if (targetClient) {
                targetClient.ws.send(
                  JSON.stringify({
                    type: "initial-data",
                    data: data.data,
                  })
                );
              }
            } else {
              // không có "to" => gửi cho toàn bộ
              for (const client of roomClients) {
                client.ws.send(
                  JSON.stringify({
                    type: "initial-data",
                    data: data.data,
                  })
                );
              }
            }
          }
          break;
        }

        case "disconnect": {
          if (rooms[data.room]) {
            for (let client of rooms[data.room]) {
              if (client.ws === ws) {
                rooms[data.room].delete(client);
                break;
              }
            }
            if (rooms[data.room].size === 0) delete rooms[data.room];
          }

          if (keyRooms[data.key]) {
            for (let client of keyRooms[data.key]) {
              if (client.ws === ws) {
                keyRooms[data.key].delete(client);
                break;
              }
            }
            if (keyRooms[data.key].size === 0) delete keyRooms[data.key];
          }
          break;
        }

        default:
          console.log("Unknown message type:", data.type);
      }
    } catch (err) {
      console.error("Error parsing message:", err);
    }
  });

  ws.on("close", () => {
    if (currentRoom && rooms[currentRoom]) {
      for (let client of rooms[currentRoom]) {
        if (client.ws === ws) {
          rooms[currentRoom].delete(client);
          break;
        }
      }
      if (rooms[currentRoom].size === 0) delete rooms[currentRoom];

      if (rooms[currentRoom]) {
        const remaining = Array.from(rooms[currentRoom]).map((c) => c.id);
      }
    }
  });
});

// Server chạy
const PORT = process.env.PORT || 8085;
const LOCAL_IP = getLocalIP();

server.listen(PORT, () => {
  const protocol = isProduction ? "https" : "http";
  console.log(`🚀 Server running on ${protocol}://localhost:${PORT}`);
  console.log(`🌐 Network: ${protocol}://${LOCAL_IP}:${PORT}`);
  console.log(`🔗 Ngrok: https://201a1d38136f.ngrok-free.app  `);
  if (isProduction) {
    console.log(`🔗 Production: https://bigapptech.vn`);
  }
  console.log(
    `📝 Environment: ${
      isProduction ? "Production (HTTPS)" : "Development (HTTP)"
    }`
  );
});
